import React, { useState, useEffect, useRef } from "react";
import Head from "next/head";
import { useTranslation } from "next-i18next";
import { serverSideTranslations } from "next-i18next/serverSideTranslations";
import {
  Infinity as InfinityIcon,
  LightningChargeFill,
  Lock,
  HeartFill,
  AwardFill,
  ShieldFillCheck,
  Check2Circle,
  ExclamationTriangle,
} from "react-bootstrap-icons";
import useUploadStats from "../hooks/useUploadStats";
import useDocuments from "../hooks/useDocuments";
import useToolsData from "../hooks/useToolsData";
import usePassword from "../hooks/usePassword";
import ProcessingFilesFormStep from "../components/ProcessingFilesFormStep";
import UploadAreaFormStep from "../components/UploadAreaFormStep";
import Steps from "../components/Steps";
import Features from "../components/Features";
import Share from "../components/Share";
import DownloadFilesFormStep from "../components/DownloadFilesFormStep";
import AvailableTools from "../components/AvailableTools";
import PasswordForm from "../components/PasswordForm";
import styles from "../styles/UploadContainer.module.css";
import {
  handleFileSelection,
  saveNewFiles,
  uploadFiles,
  downloadFiles,
} from "../helpers/utils.js";
import Alerts from "../components/Alerts";
import pageStyles from "../styles/Page.module.css";

export async function getStaticProps({ locale }) {
  return {
    props: {
      ...(await serverSideTranslations(locale, ["common", "protect-pdf"])),
    },
  };
}
const ProtectPDFPage = () => {
  const { ProtectPDFTool } = useToolsData();
  const mountedRef = useRef(false);
  const [isSpinnerActive, setIsSpinnerActive] = useState(false);
  const [formStep, setFormStep] = useState(0);
  //loadedFilesCount is used to count the files currently being loaded to show spinner while loading the files
  const [loadedFilesCount, setLoadedFilesCount] = useState(0);
  const [requestSignal, setRequestSignal] = useState();
  const { t } = useTranslation();

  const {
    resultsInfoVisibility,
    resultsErrors,
    handleResetInitialUploadState,
    handleUpdateResultsDisplay,
  } = useUploadStats();

  const {
    documents,
    handleAddDocument,
    handleUpdateDocument,
    handleResetInitialDocumentsState,
  } = useDocuments();

  const {
    password,
    confirmPassword,
    passwordsMatch,
    handlePasswordChange,
    handleConfirmPasswordChange,
    handleResetPassword,
  } = usePassword();

  const handleChange = (event) => {
    //Calling handleFileSelection function to extract pdf pages and their data and insert them in an array
    handleFileSelection(
      event,
      setLoadedFilesCount,
      handleAddDocument,
      t,
      mountedRef,
      ProtectPDFTool
    );
  };

  const handleHandleResetInitialStates = () => {
    handleResetInitialDocumentsState();
    handleResetInitialUploadState();
    handleResetPassword();
    setFormStep(0);
  };

  const handleDownload = () => {
    saveNewFiles(documents);
  };

  useEffect(() => {
    //set mountedRef to true
    mountedRef.current = true;
    //Axios AbortController to abort requests
    const controller = new AbortController();
    const signal = controller.signal;
    setRequestSignal(signal);
    //cleanup function
    return () => {
      // cancel all the requests
      controller.abort();
      //set mountedRef to false
      mountedRef.current = false;
    };
  }, []);

  useEffect(() => {
    // if loadedFilesCount (count of file currently being loaded) is greater than zero than show spinner
    if (loadedFilesCount > 0) {
      //show spinner
      if (mountedRef.current) {
        setIsSpinnerActive(true);
      }
    } else {
      //after all files are loaded, hide spinner
      if (mountedRef.current) {
        setIsSpinnerActive(false);
      }
    }
  }, [loadedFilesCount]);

  const handleSubmit = async (e) => {
    e.preventDefault();

    /**
     * File protection  will be done on two steps:
     *** First step : uploading file to server and start processing it
     *** Second step : sending periodic download requests to check if file is done processing and return the result.
     */

    //updating form step in UI
    setFormStep(2);

    //First step : Uploading Files & Start Files Processing
    // Storing password in data Array-like object
    const data = {
      password: password,
    };

    const { uploadResponsesArray, uploadResponsesUnsuccessfulRequests } =
      await uploadFiles({
        signal: requestSignal,
        documents: documents,
        uri: ProtectPDFTool.URI,
        data: data,
      });

    //in case error occurred while uploading file
    if (uploadResponsesUnsuccessfulRequests.length === 1) {
      handleUpdateResultsDisplay(false, uploadResponsesUnsuccessfulRequests);
      setFormStep(3);
      return;
    }

    //Second step : Check if files are done processing
    const { downloadResponsesArray, downloadResponsesUnsuccessfulRequests } =
      await downloadFiles({
        responseMimeType: ProtectPDFTool.outputFileMimeType,
        signal: requestSignal,
        uploadResponsesArray,
        handleUpdateDocument,
      });

    //check if all documents have been processed, no failed documents
    if (downloadResponsesArray.length === 1) {
      handleUpdateResultsDisplay(true, []);
    } else {
      //in case error occurred while downloading the file
      //check if document has failed being processed
      handleUpdateResultsDisplay(false, downloadResponsesUnsuccessfulRequests);
    }
    //updating form step in UI
    setFormStep(3);
  };

  useEffect(() => {
    if (documents.length <= 0) {
      setFormStep(0);
    } else {
      setFormStep(1);
    }
  }, [documents.length]);

  return (
    <>
      <Head>
        {/* Anything you add here will be added to this page only */}
        <title>{t("protect-pdf:title")}</title>
        {/* IMPORTANT: Customize the meta tags below for better SEO. 
            - "keywords" should include relevant terms users might search for (e.g., PDF tools, merge PDF, compress PDF). 
            - "description" should be a brief, compelling summary of your site (e.g., "A powerful online tool for managing PDFs – merge, compress, convert, and more!"). 
            - Avoid duplicate content across multiple websites.
            - The "description" and "keywords" values can be modified in the respective JSON files for each page.  
            - You can find these files in the `public/locales` folder.  
            - For example, to update the meta tags for the "protect-pdf" page, edit the values of "meta_description" and "meta_keywords" in the `protect-pdf.json` file inside each locale folder.   */}
        <meta name="description" content={t("protect-pdf:meta_description")} />
        <meta name="Keywords" content={t("protect-pdf:meta_Keywords")} />
        {/* You can add your canonical link here to override the one in _app.js */}
        {/* You can add your alternate links here, example: */}
        <link
          rel="alternate"
          href={`${process.env.NEXT_PUBLIC_APP_URL}${ProtectPDFTool.href}`}
          hrefLang="en"
        />
        <link
          rel="alternate"
          href={`${process.env.NEXT_PUBLIC_APP_URL}/es${ProtectPDFTool.href}`}
          hrefLang="es"
        />
        <link
          rel="alternate"
          href={`${process.env.NEXT_PUBLIC_APP_URL}/ar${ProtectPDFTool.href}`}
          hrefLang="ar"
        />
        <link
          rel="alternate"
          href={`${process.env.NEXT_PUBLIC_APP_URL}/zh${ProtectPDFTool.href}`}
          hrefLang="zh"
        />{" "}
        <link
          rel="alternate"
          href={`${process.env.NEXT_PUBLIC_APP_URL}/de${ProtectPDFTool.href}`}
          hrefLang="de"
        />
        <link
          rel="alternate"
          href={`${process.env.NEXT_PUBLIC_APP_URL}/fr${ProtectPDFTool.href}`}
          hrefLang="fr"
        />
        <link
          rel="alternate"
          href={`${process.env.NEXT_PUBLIC_APP_URL}/it${ProtectPDFTool.href}`}
          hrefLang="it"
        />
        <link
          rel="alternate"
          href={`${process.env.NEXT_PUBLIC_APP_URL}/pt${ProtectPDFTool.href}`}
          hrefLang="pt"
        />
        <link
          rel="alternate"
          href={`${process.env.NEXT_PUBLIC_APP_URL}/ru${ProtectPDFTool.href}`}
          hrefLang="ru"
        />
        <link
          rel="alternate"
          href={`${process.env.NEXT_PUBLIC_APP_URL}/uk${ProtectPDFTool.href}`}
          hrefLang="uk"
        />
        <link
          rel="alternate"
          href={`${process.env.NEXT_PUBLIC_APP_URL}/id${ProtectPDFTool.href}`}
          hrefLang="id"
        />
        <link
          rel="alternate"
          href={`${process.env.NEXT_PUBLIC_APP_URL}/da${ProtectPDFTool.href}`}
          hrefLang="da"
        />
        <link
          rel="alternate"
          href={`${process.env.NEXT_PUBLIC_APP_URL}/nl${ProtectPDFTool.href}`}
          hrefLang="nl"
        />
        <link
          rel="alternate"
          href={`${process.env.NEXT_PUBLIC_APP_URL}/hi${ProtectPDFTool.href}`}
          hrefLang="hi"
        />
        <link
          rel="alternate"
          href={`${process.env.NEXT_PUBLIC_APP_URL}/ko${ProtectPDFTool.href}`}
          hrefLang="ko"
        />
        <link
          rel="alternate"
          href={`${process.env.NEXT_PUBLIC_APP_URL}/ja${ProtectPDFTool.href}`}
          hrefLang="ja"
        />
      </Head>

      <main>
        <header className="page_section header mb-0">
          <h1 className="title">{t("protect-pdf:page_header_title")}</h1>
          <p className="description">{t("protect-pdf:page_header_text")}</p>
        </header>

        <section className="page_section mt-0">
          <article className="container">
            <section className={pageStyles.tool_container_wrapper}>
              {/* Container start */}
              {formStep === 0 && (
                <UploadAreaFormStep
                  handleChange={handleChange}
                  isSpinnerActive={isSpinnerActive}
                  isMultipleInput={false}
                  acceptedMimeType={ProtectPDFTool.acceptedInputMimeType}
                />
              )}
              {formStep === 1 && (
                <PasswordForm
                  password={password}
                  confirmPassword={confirmPassword}
                  passwordsMatch={passwordsMatch}
                  setPassword={handlePasswordChange}
                  setConfirmPassword={handleConfirmPasswordChange}
                  handleSubmit={handleSubmit}
                  actionTitle={t("protect-pdf:protect_pdf")}
                />
              )}

              {formStep === 2 && (
                <ProcessingFilesFormStep
                  progress={t("protect-pdf:locking_pdf")}
                />
              )}

              {formStep === 3 && (
                <DownloadFilesFormStep
                  title={
                    documents.length === 1
                      ? t("common:your_document_is_ready")
                      : documents.length > 1
                      ? t("common:your_documents_are_ready")
                      : ""
                  }
                  handleDownload={handleDownload}
                  handleResetInitialState={handleHandleResetInitialStates}
                >
                  {resultsInfoVisibility && (
                    <div className="row w-100 d-flex justify-content-center text-center mt-5 mb-5">
                      <Check2Circle size={130} color="#7d64ff" />
                    </div>
                  )}
                  {resultsErrors.length > 0 && (
                    <Alerts
                      message={t("common:files_error")}
                      filesData={resultsErrors}
                      type="error"
                      icon={<ExclamationTriangle size={22} />}
                    />
                  )}
                </DownloadFilesFormStep>
              )}
              {/* Container end */}
            </section>
          </article>
        </section>

        {/* steps Start */}
        <Steps
          title={t("protect-pdf:how_to_title")}
          stepsArray={[
            {
              number: 1,
              description: t("protect-pdf:how_to_step_one"),
            },
            {
              number: 2,
              description: t("protect-pdf:how_to_step_two"),
            },
            {
              number: 3,
              description: t("protect-pdf:how_to_step_three"),
            },
          ]}
        />
        {/* steps end */}
        {/* features start */}
        <Features
          title={t("common:features_title")}
          featuresArray={[
            {
              title: t("protect-pdf:feature_one_title"),
              description: t("protect-pdf:feature_one_text"),
              icon: <LightningChargeFill />,
            },
            {
              title: t("protect-pdf:feature_two_title"),
              description: t("protect-pdf:feature_two_text"),
              icon: <InfinityIcon />,
            },
            {
              title: t("protect-pdf:feature_three_title"),
              description: t("protect-pdf:feature_three_text"),
              icon: <Lock />,
            },
            {
              title: t("protect-pdf:feature_four_title"),
              description: t("protect-pdf:feature_four_text"),
              icon: <ShieldFillCheck />,
            },
            {
              title: t("protect-pdf:feature_five_title"),
              description: t("protect-pdf:feature_five_text"),
              icon: <HeartFill />,
            },
            {
              title: t("protect-pdf:feature_six_title"),
              description: t("protect-pdf:feature_six_text"),
              icon: <AwardFill />,
            },
          ]}
        />
        {/* features end */}
        {/* Article Start */}
        <section className="page_section">
          <article className={`container ${pageStyles.article_section}`}>
            <header className={pageStyles.article_header}>
              <h2 className={pageStyles.title_section}>
                {t("protect-pdf:article_title")}
              </h2>
              <div
                className={`${pageStyles.divider} ${pageStyles.mx_auto}`}
              ></div>
            </header>

            <section className={pageStyles.article_content}>
              <p className="get_app">
                The source code of this application, available under licenses,
                can be purchased at{" "}
                <a href="https://codecanyon.net/item/pdf-tools-high-quality-pdf-tools-nextjs-react-web-application/44924651">
                  https://codecanyon.net/item/pdf-tools-high-quality-pdf-tools-nextjs-react-web-application/44924651
                </a>
              </p>
              <p>{t("protect-pdf:article_paragraph_01")}</p>
              <p>{t("protect-pdf:article_paragraph_02")}</p>
              <p>{t("protect-pdf:article_paragraph_03")}</p>
            </section>
          </article>
        </section>
        {/* Article End */}
        <AvailableTools />
        <Share />
      </main>
    </>
  );
};
export default ProtectPDFPage;
